<?php

namespace App\Controller;

use DateTime;
use Exception;
use App\Entity\Note;
use App\Entity\NoteJson;
use App\Repository\NoteRepository;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

/**
 * @Route("/note")
 * @IsGranted("ROLE_EDITOR")
 */
class NoteController extends AbstractController
{
    /**
     * @Route("/", name="note_index", methods={"GET"})
     */
    public function index(NoteRepository $noteRepository): Response
    {
        return $this->render('note/index.html.twig', [
            'notes' => $noteRepository->findAll(),
        ]);
    }

    // https://127.0.0.1:8000/note/list/-editor-apprenant-*-4
    /**
     * @Route("/list/{route}", methods={"GET"})
     */
    public function liste($route, NoteRepository $noteRepository): Response
    {
        $route = str_replace('-','/',$route); //dd($route);
        $notes = $noteRepository->findBy(['route'=>$route], ['created'=>'desc']);
        return $this->json($notes,200,[],['groups'=>'get:read']);
    }


    // https://127.0.0.1:8000/note/new
    /**
     * @Route("/new", methods={"POST"})
     */
    public function new(Request $request,
                        SerializerInterface $serializer, 
                        ValidatorInterface $validator): Response
    {
        $jsonRecu = $request->getContent();     
        try {
            //désérialisation du json
            $jsonNote = $serializer->deserialize($jsonRecu, NoteJson::class, 'json');           

            //validation des propriétés
            $errors = $validator->validate($jsonNote);
            if(count($errors)>0) {
                return $this->json(['status'=>400, 'message'=>"Données d'entrée incorrectes"], 400);
            }    
                        
            //désérialisation OK : création de la note 
            $note = new Note();
            $note->setTexte( $jsonNote->getTexte() );
            $note->setRoute( $jsonNote->getRoute() );
            $note->setCreated(new DateTime());
            $user = $this->getUser();
            $note->setAuteur( $user );

            $entityManager = $this->getDoctrine()->getManager();
            $entityManager->persist($note);
            $entityManager->flush();              

            //retour de la note créée
            return $this->json($note,200,[],['groups'=>'get:read']);

        } catch(Exception $e) {
            return $this->json(['status'=>400, 'message'=> 'Error: '.$e->getMessage()], 400);
            //return $this->json(['status'=>400, 'message'=>"Données d'entrée incorrectes"], 400);
        }

        return $this->json(['status'=>400, 'message'=>"Données d'entrée incorrectes"], 400);
    }

    /**
     * @Route("/{id}", methods={"DELETE"})
     */
    public function delete(Request $request, Note $note): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $entityManager->remove($note);
        $entityManager->flush();
        return $this->json([],200,[],[]);
    }
}
