<?php

namespace App\Controller;

use App\Entity\Apprenant;
use App\Entity\Resultats;
use App\Entity\PromoAppre;
use App\Form\ApprenantType;
use App\Service\FileUploader;
use App\Repository\ApprenantRepository;
use App\Repository\FormationRepository;
use App\Repository\PromotionRepository;
use App\Repository\ResultatsRepository;
use App\Repository\CompetenceRepository;
use App\Repository\EvaluationRepository;
use App\Repository\PromoAppreRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Config\Definition\Exception\Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;

class EditApprenantController extends AbstractController
{

    /**
     * afficher les apprenants avec une pagination 
     * 
     * @Route("/editor/apprenant/{page<\d+>?0}", name="editor_apprenant_liste")
     */
    public function apprenant_liste(Request $request, ApprenantRepository $apprenantRepository, PromotionRepository $promotionRepository, $page)
    { 
        //liste des propos
        $promos = $promotionRepository->findBy([], ['DateDebut' => 'desc']);

        // récupérer l'input chercher et la promotion
        $nom = $request->get('chercher_apprenant');
        $promo = $request->get('chercher_promo');   
        if($promo==null && $page==0 ) {
            //seulement si on n'accède pas via la pagination, dans ce cas c'est tout!
            $promo=$promos[0]->getId();
        }
        if($page==0) $page=1; //pour la suite des calculs        

        if($promo<1 && strlen($nom)<1) {
            //tous les apprenants !
            $limit = 10;
            
            $start = $page * $limit - $limit;
            $all = count($apprenantRepository->findAll());
            $pages = ceil($all / $limit);
            $apprenants = $apprenantRepository->findBy([], ['Nom' => 'asc'], $limit, $start);
        }
        else {
            //une promo en particulier ou une recherche par nom
            $apprenants = $apprenantRepository->findAllByPromoAndNom($nom, $promo);
            $all = count($apprenants);
            $pages = 1;
        }
        
        return $this->render('editor/apprenant/apprenant_liste.html.twig', [
            'apprenants' => $apprenants,
            'pages' => $pages,
            'page' => $page,
            'promos' => $promos,
            'promoid' => $promo,
            'nbtotal' => $all,
        ]);
    }

    /**
     * chercher un apprenant par son nom ou prénom
     * @Route("/editor/apprenant/chercher/{page<\d+>?1}", name="editor_apprenant_chercher")
     */
    public function chercher(Request $request, ApprenantRepository $repo, PromotionRepository $promotionRepository, $page)
    {

        // récupérer l'input chercher et la promotion
        $nom = $request->get('chercher_apprenant');
        $promo = $request->get('chercher_promo');

        if(strlen($promo)<1 && strlen($nom)<1) {
            $limit = 10;
            $start = $page * $limit - $limit;
            $all = count($repo->findAll());
            $pages = ceil($all / $limit);
            $apprenants = $repo->findBy([], ['Nom' => 'asc'], $limit, $start);
        }
        else {
            $apprenants = $repo->findAllByPromoAndNom($nom, $promo);
            $limit = 10;
            $start = $page * $limit - $limit;
            $all = count($apprenants);
            $pages = ceil($all / $limit);
        }

        //liste des propos
        $promos = $promotionRepository->findBy([], ['DateDebut' => 'desc']);

        return $this->render('editor/apprenant/apprenant_liste.html.twig', [
            'apprenants' => $apprenants,
            'pages' => $pages,
            'page' => $page,
            'promos' => $promos,
            'promoid' => $promo,
            'nbtotal' => $all,
        ]);
    }


    /**
     * afficher un apprenant
     * 
     * @Route("/editor/apprenant/show/{id}", name="editor_apprenant_show")
     */
    public function apprenant_show(Apprenant $apprenant, PromoAppreRepository $promoAppreRepository, 
                                   CompetenceRepository $competenceRepository)
    {
        //promotion en cours de l'apprenant -> fiche profil selon compétences de la formation
        $ficheprofil = [];
        $evalsdata = [];
        $promoAppres = $apprenant->getPromoAppres();
        if(count($promoAppres)>0) {
            $derpromoAppr = $promoAppres[count($promoAppres)-1];
            $derpromotion = $derpromoAppr->getPromotion();
            $ficheprofil = $competenceRepository->findByCompetenceForApprenant($derpromotion->getId(), $derpromoAppr->getid()); //dd($ficheprofil);
            $evalsdata = $competenceRepository->findByEvaluationForApprenant($derpromotion->getId(), $derpromoAppr->getid());  //dd($evalsdata);
        }
        return $this->render('editor/apprenant/apprenant_show.html.twig', [
            'apprenants' => $promoAppreRepository->findBy(['apprenant' => $apprenant]),
            'ficheprofil'=>$ficheprofil,
            'evalsdata'=>$evalsdata,
        ]);
    }

    /**
     * ajouter un nouveau apprenant et créer un nouveau utilisateur dans User class
     * 
     * @Route("/editor/apprenant/new", name="editor_apprenant_new")
     */
    public function apprenant_new(Request $request, UserPasswordEncoderInterface $encoder, FileUploader $fileUploader)
    {
        $apprenant = new Apprenant();
        $manager = $this->getDoctrine()->getManager();

        $form = $this->createForm(ApprenantType::class, $apprenant);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            // récupérer les réseaux ajoutés et les insérer dans la table réseau
            foreach ($apprenant->getReseaux() as $reseau) {
                $reseau->setApprenant($apprenant);
                $manager->persist($reseau);
            }

            // ici on récupère le fichier téléversé.et on teste s'il existe ou non.
            // s'il existe, on appelle le service FileUploader et on stock le fichier 
            // dans une variable pour le stocker dans l'apprenent en question
            $brochureFile = $form['brochure']->getData();
            if ($brochureFile) {
                $brochureFileName = $fileUploader->upload($brochureFile);
                $apprenant->setAvatar($brochureFileName);
            }

            // créer un mot de passe comboné du nom et du prénom et le crypter
            $mdp = strtolower($apprenant->getNom() . $apprenant->getPrenom());

            $mdp_hash = $encoder->encodePassword($apprenant, $mdp);

            $apprenant->setPassword($mdp_hash);

            try {

                $manager->persist($apprenant);
                $manager->flush();

                $this->addFlash('success', "L'apprenant - {$apprenant->getFullname()} - a été ajouté!");
                return $this->redirectToRoute('editor_apprenant_liste');
            } catch (Exception $e) {
                $this->addFlash('danger', 'Cet email existe déjà!');
            }
        }

        return $this->render('editor/apprenant/appreant_new.html.twig', [
            'form' => $form->createView()
        ]);
    }





    /**
     * modifier un apprenant
     * 
     * @Route("/editor/apprenant/edit/{id}", name="editor_edit_apprenant")
     */
    public function edit_apprenant(Apprenant $newApprenant, Request $request, FileUploader $fileUploader)
    {

        $manager = $this->getDoctrine()->getManager();

        $form = $this->createForm(ApprenantType::class, $newApprenant);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {


            // récupérer les réseaux ajoutés et les insérer dans la table réseau
            foreach ($newApprenant->getReseaux() as $reseau) {
                $reseau->setApprenant($newApprenant);
                $manager->persist($reseau);
            }


            $brochureFile = $form['brochure']->getData();

            if ($brochureFile) {
                $brochureFileName = $fileUploader->upload($brochureFile);
                $newApprenant->setAvatar($brochureFileName);
            }

            $manager->flush();

            $this->addFlash("warning", "L'apprenant {$newApprenant->getFullname()} a été modifié!");
            return $this->redirectToRoute('editor_apprenant_liste');
        }
        return $this->render('editor/apprenant/apprenant_edit.html.twig', [
            'form' => $form->createView(),
            'apprenant' => $newApprenant
        ]);
    }


    /**
     * supprimer un apprenant
     * @Route("/editor/apprenant/delete/{id}", name="editor_apprenant_delete")
     */
    public function delete_apprenant(Apprenant $apprenant, EntityManagerInterface $manager)
    {
        $manager->remove($apprenant);
        $manager->flush();
        $this->addFlash("danger", "L'apprenant {$apprenant->getFullname()} a été supprimé !");

        return $this->redirectToRoute('editor_apprenant_liste');
    }



    /**
     * gestion de competence
     * 
     * @Route("/editor/competence", name="editor_competence")
     */
    public function competence(ApprenantRepository $repo)
    {

        return $this->render('editor/apprenant/competence.html.twig', []);
    }
}
